;; define global variables, patches, and foragers
extensions [ gis time]
globals [ tides sunlight max-biomass total-biomass elevation-dataset max-height min-height 
  current-time t-stamp tide-time-series sunlight-time-series tide-patches census1 census2 census3
  lowroosttotal midroosttotal hiroosttotal ] 
breed [curlews curlew]  
turtles-own [energy weight efficiency dominance roosted]       
patches-own [biomass heights census nearest-neighbors]


to setup
  clear-all
  
  ;; using the gis extension
  ;; upload coordinate system and any gis datasets  
    
  if scenario = 1[  set elevation-dataset gis:load-dataset "HBayNoSLR10_19.asc"]
  
  if scenario = 2[ set elevation-dataset gis:load-dataset "MinSLRGridOut10_23.asc"]
  
  if scenario = 3[set elevation-dataset gis:load-dataset "MeanSLRGridOut10_23.asc"]
  
  if scenario = 4[ set elevation-dataset gis:load-dataset "MaxSLRGridOut10_23.asc"]
 
 
 ;;set the elevation heights  for each patch
  gis:apply-raster elevation-dataset heights
  ask patches [
    set heights precision heights 4        ;;;lowers the number of decimal places
  ]
  
     
  ;;set patch variables
  ;;for visual reference, patch color based on elevation`
  ;;default intertidal range: -0.6 < h < 2
  ;; min- and max-height may be set as sliders or replaced with constants instead
  
  set min-height -0.6
  set max-height 2
  
  ask patches [
     set biomass 0
     set census 0
          
     if heights < max-height [ set pcolor 46 ]          ;tidal/ocean
     if heights < 0 [ set pcolor 44 ]                   ;;intertidal range
     if heights < min-height [ set pcolor 95 ]          ;;shallower water
     if heights < -10 [ set pcolor 92 ]        ;;deep water
     if heights >= 2 [ set pcolor 55.5 ]       ;;roost elevation
     if heights > 3.5 [ set pcolor 54.5 ]      ;;low elevation
     if heights >= 6 [ set pcolor 53.5 ]       ;;med+ elevation
    ]   
  
  ;;intertidal habitat patch settings
  ask patches [
    if (heights > min-height) and (heights < max-height) [ set biomass initial-biomass ]     ;;intertidal habitat elevations
    
    set nearest-neighbors patches with 
    [(distance myself < 9)  and (heights > min-height) and (heights < max-height) ]     ;;;;note change coefficient to max range
  ]
    
  set tide-patches patches with [(heights >= min-height) and (heights <= max-height)]
  
  ;;note: setting neighbors and tide-patches makes the code procedures focus only on intertidal habitat patches 
  ;;considering patches that are not within the habitat range will increase calculations and processing time
    
      
    
  ;;create the curlews and set their variables
  set-default-shape curlews "curlew"
  create-ordered-curlews initial-number-curlews  
  [
    set color 36.5
    set size 1  
    set label-color blue - 2
    set weight random (100) + 650
    set energy weight * 33.7
    
    set efficiency 0.8 + random-float 0.4
    set dominance (initial-number-curlews - who ) / initial-number-curlews
    
    set roosted 1
    
  ]
  
    
  ;; distribute curlews in initialization to ensure birds do not clump or start too far from habitat
  ;; when patches have same initial-biomass, curlews choose intertidal patches with least occupants
  set tides 0.6
  foreach curlew-list [
    ask ?[
      let best-patch max-one-of first-patch [score]
      move-to best-patch
      ;;an error here during setup can be fixed by playing with initial-curlew-number or max-curlew parameters
      ;;What do you think this setup error represents?
    ]]



  ;; display-labels ;;displays energy of curlews and biomass of patches, Note: can be clunky and hard to read

  ;; using the time
  set current-time time:anchor-to-ticks (time:create "1998-09-01 00:00") 1 "hour"
  set tide-time-series time:ts-load "tidesSep1_Mar311998.csv"
  set sunlight-time-series time:ts-load "sunlightSep1_Mar311998.csv"

  reset-ticks
      
  ;;set census1 0  
  ;;set census2 0
  ;;set census3 0
    
  
end



to go
  if (not any? turtles) or (ticks > 5088) [ stop ]
  
  ;;;;Part 1 of tick: update patches vs. tides and distribute curlews to habitat
  ebb-and-flow
  distribute
  
  
  ;;;;Part 2: curlews eat and metabolise
  foreach curlew-list [
    ask ?[
    eat-biomass
    death
  ]]
  
  
  
  ;;;;Part 2.5: census curlews on obs dates for output in experiment
  
  ;;if ticks = 808 [ set census1 count curlews ]
  
  ;;if ticks = 3641 [ set census2 count curlews   ]
  
  ;;if ticks = 4312 [  set census3 count curlews  ]
  
  
  
  ;;;;Part 3: updates and outputs
  
  ;;biomass-deplete    ;;can be used if biomass has submodel separate from foraging (dessication or seasonal)
  
  calc-total-biomass   ;;used for a graph or for output
  
  if show-labels[ display-labels]      ;;use to display biomass, weight
  
  show-sun             ;;a visual reference for whether sun is up (sunlight=1) during that hour
  
  
  
  ;;;;Part 4: Finish up
  tick
  
  output-print time:show current-time "MMMM d, yyyy HH:mm"
  
    
    
end




;;;;;;;;;;;;;;;;;;;PROCEDURES;;;;;;;;;;;;;;;;;;;


;;Part 1 Procedures

;;update tide height and sunlight values
;;update visual reference to whether intertidal patches are available

to ebb-and-flow          
  set tides time:ts-get tide-time-series current-time "tide-elevation"
  set sunlight time:ts-get sunlight-time-series current-time "light-out"
  ask tide-patches [
     ifelse (heights > -0.8) and (heights <= tides) [
       set pcolor 44.5
       ]
     [if (heights > tides) and (heights < 2.1) [ set pcolor 46 ]
     ]]   
    
end


;;curlews choose patches to feed or roost on by
;;comparing available patches within their foraging range
;;if sunlight = 0, all curlews roost
;;else, curlews choose from available foraging patches in order

to distribute
  ifelse sunlight = 0[
    ask curlews[
     let resting-patch min-one-of roost-patches [distance myself]
     move-to resting-patch  
  ]] 
  [  
  foreach curlew-list [
    ask ?[
      let best-patch max-one-of available-patches [score]
      
    ifelse best-patch != nobody [ move-to best-patch ]
       [ let resting-patch min-one-of roost-patches [distance myself]
         move-to resting-patch 
          ]
  ]]
  ]
end

;;reporters involved with distribution

to-report curlew-list
  report sort curlews
end 


to-report available-patches
  let this-curlew self
  report (nearest-neighbors) with 
  [(count turtles-here with [self != this-curlew] < max-curlews) and ( heights < 2 ) and (heights > tides)]
end


to-report roost-patches
  report patches with [ (heights > 2) and (heights < 8) ]
end

to-report first-patch
  report patches with [(count turtles-here < max-curlews) and (heights < 2) and (heights > tides)]
end

to-report score
  report biomass * (0.75 + random-float 0.5)      
end
;;the random value represents an "error term", i.e. birds estimate patch quality/abudance





;;Part 2 Procedures

;;curlews eat biomass, metabolise, and reduce patch biomass

to eat-biomass   
     ifelse (biomass >= 1) and (weight < max-weight) [
     
     set energy energy + IR * 22          ;;;energy = energy + intake rate(g)  * 22 kj /g (energy per g of food)
     let depletion IR
     
     
     
     set biomass biomass - depletion 
     metabolise
     set roosted 0
     ]  
     [ 
       roost 
       set roosted 1
       ]

end

to metabolise
  let bmr 3.6 * 5.06 * ( weight / 1000 ) ^ 0.729  ;bmr equation is in watts -> *3.6 converts to kj/hr
  let dee 6 * bmr                                 
  set energy energy - dee  
  set weight energy / 33.7
end

to roost
  let bmr 3.6 * 5.06 * ( weight / 1000 ) ^ 0.729
  let dee bmr
  set energy energy - dee  
  set weight energy / 33.7
end



;;death procedure may represent both starvation and emigration from the system

to death  ;; when energy drops below zero, curlew emigrates
  if weight < 400 [ 
    die 
    ]
end
  

;;;;;;;;;;;;;MORPH equations for intake rates (functional response) 

to-report IFIR  
  report efficiency * IFIRmax * ( biomass / ( biomass + 0.761 ))   
end

;;IFIRmax can be estimated in different ways, e.g. converting units of max observed intake or a statistical model
 
;;Late Mistake: B50=0.761 needed to be scaled to match desired units. Be careful with the units in the intake equations.
;;Given its small value compared to actual biomass values, the value B50 was scaled out and had an insignificant effect
;;This means I used a "simplified" IFIR, where the ( biomass / ( biomass + 0.761 )) term is a dummy variable near equal to 1
;;and that the IFIR equation is mostly a function of IFIRmax and the individual's efficiency.
;;The B50=0.761 is valuable if data on prey distributions and densities are available and are not generalized as "biomass"
;;and if competition interference plays a stronger role in occupancy and survival/emigration.


to-report curlew-density 
  report count curlews-here  
end

to-report IR
  ifelse ( curlew-density > 4) 
  [ report IFIR * ( curlew-density / 4 ) ^ (-( 0.08 - (0.08) * dominance)) ] 
  [ report IFIR]
end








;;;;;;;;;;;;;; Tools for monitoring analyzing and questioning

to show-sun   ;;used as a visual reference
  ifelse sunlight = 0[
  ask patch 3 46 [ set pcolor 42 ]
  ask patch 3 47 [ set pcolor 42 ]
  ask patch 4 46 [ set pcolor 42 ]
  ask patch 4 47 [ set pcolor 42 ]
  ]
  [
  ask patch 3 46 [ set pcolor 45 ]
  ask patch 3 47 [ set pcolor 45 ]
  ask patch 4 46 [ set pcolor 45 ]
  ask patch 4 47 [ set pcolor 45 ]
  ] 
end


to calc-total-biomass ;;used graphically to see any trends in biomass levels
  set total-biomass sum [biomass] of patches
end


to display-labels
   ask curlews [ set label round weight ] 
   ask patches [ set plabel round biomass ]
end



;;Procedures for Buttons

to census-curlews
  ask patches 
  [ set census count curlews-here ]

end 

to census-map
  ask patches[
    if census >= 15 [set pcolor 0 ]
    if census < 15 [ set pcolor 3 ]
    if census <= 7 [ set pcolor 5 ]
    if census < 1 [ set pcolor 9.9 ]  
  ]
end


to biomass-diagnose
  ask patches with [(heights > min-height ) and (heights < max-height)]
  [set pcolor scale-color red biomass initial-biomass 0]
end
  
  
;;Tools for graphing and asking questions
 
to-report average-weight
  report mean [weight] of curlews 
end

to-report biomasscons
  report mean [biomass] of tide-patches / initial-biomass
end
  
  
  
  
@#$#@#$#@
GRAPHICS-WINDOW
427
10
877
591
-1
-1
11.0
1
10
1
1
1
0
0
0
1
0
39
0
49
1
1
1
ticks
10.0

SLIDER
8
89
176
122
initial-number-curlews
initial-number-curlews
0
350
215
1
1
NIL
HORIZONTAL

BUTTON
15
16
84
49
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
104
18
171
51
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
26
249
231
391
Total Curlew Population
time
population
0.0
1.0
0.0
6.0
true
false
"" ""
PENS
"curlews" 1.0 0 -13345367 true "" "plot count curlews"

MONITOR
207
233
258
282
curlews
count curlews
3
1
12

TEXTBOX
9
66
149
85
Parameter settings
11
0.0
0

SWITCH
272
174
408
207
show-labels
show-labels
1
1
-1000

PLOT
252
256
412
384
biomass levels
value
occurence
0.0
100.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -12087248 true "" "plot total-biomass"

SLIDER
183
90
355
123
initial-biomass
initial-biomass
5000
30000
18000
500
1
NIL
HORIZONTAL

BUTTON
292
485
399
518
census-map
census-curlews\ncensus-map
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
8
130
180
163
max-sight
max-sight
0
6
3
1
1
NIL
HORIZONTAL

OUTPUT
638
479
901
565
15

SLIDER
187
129
359
162
max-curlews
max-curlews
0
20
10
1
1
NIL
HORIZONTAL

PLOT
22
411
271
592
weight of four curlews
time
weight
0.0
10.0
300.0
820.0
true
false
"" ""
PENS
"weight1" 1.0 0 -13840069 true "" "if turtle 0 != nobody [plot [weight] of turtle 0]"
"pen-1" 1.0 0 -13791810 true "" "if turtle 100 != nobody [plot [weight] of turtle 100]"
"pen-2" 1.0 0 -2674135 true "" "if turtle 280 != nobody [plot [weight] of turtle 280]"
"average" 1.0 0 -7500403 true "" "plot average-weight"
"pen-4" 1.0 0 -955883 true "" "if turtle 180 != nobody [plot [weight] of turtle 180]"

SLIDER
11
171
183
204
max-weight
max-weight
500
1000
850
20
1
NIL
HORIZONTAL

BUTTON
292
435
398
469
NIL
biomass-diagnose
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
335
541
408
601
IFIRmax
6.29
1
0
Number

CHOOSER
200
18
338
63
scenario
scenario
1 2 3 4
3

@#$#@#$#@
## WHAT IS IT?

This model explores how foraging shorebird populations can be represented within the individual-based modeling framework and how population numbers are sensitive to the conditions of their environment. In this example, we focus on the Long-billed Curlew in the intertidal habitats of Humboldt Bay, CA and how their foraging behavior depends on the availability of exposed tidal flats and other environmental factors.

Global warming and sea level rise are predicted to cause loss of intertidal habitat as dikes and human development prevent new habitat from forming. This loss of habitat may significantly change population numbers and estimating population level responses may help to increase public awareness and drive new management decisions.

## HOW IT WORKS

The agents of this model are the individual curlews that make up the population at Humboldt Bay during the non-breeding season (roughly fall to early spring). Their main goal is to make it through the winter by foraging for invertebrate prey throughout the exposed tidal flats. 
The patches in the model represent the tidal flats. Each patch has its own amount invertebrate prey, represented here as “biomass”, which is consumed by the curlews. However, the patches' biomass is not always available to the curlews and depends on two main environmental factors: the height of the tide and sunlight hours. These two factors are updated with every tick of the model (time-step = 1 hour). If patches are available, a maximum number of birds can occupy a single patch. This simulates the territorial behavior of more dominant birds. For the curlews, the preferred foraging time is daytime, during low-tide, when the most area of tidal flats are exposed and visibility of surroundings is high, e.g., in case of nearby predators or competitors.

Each patch, along with its own biomass store, is assigned a constant elevation value. This elevation value, when compared with the changing tide height, determines whether a patch can be foraged on.  When the tide height is less than the patch's elevation height, the patch becomes exposed. In contrast, when tide height is greater than the patch's elevation height, the patch becomes inundated and is unavailable for foraging. 
When there are patches available to forage on, curlews choose patches in order of dominance, where the most dominant curlew chooses first. When patches are unavailable to forage on, curlews decide to roost nearby and wait until patches become available again. Lastly, curlews who are unable to consume enough food and run out of energy reserves ultimately emigrate or starve (represented singly as the “die” procedure).
In summary, during one tick/timestep:
1. the environment (tides and sunlight) are updated
2. the curlews choose patches based on availability
3. the curlews forage or roost for the hour
4. the gains and losses in energy are calculated

Details of how the model was constructed and used for research can be found in my Thesis Research Paper, referenced below. The original forager-prey model that this model is based on is described and utilized by the other resources listed.

## HOW TO USE IT

1. Adjust the slider parameters (see below), or use the default settings. 
2. Press the SETUP button. 
3. Press the GO button to begin the simulation. 
4. Look at the monitors to see the current population sizes 
5. Play around!


Parameters:
Scenario: Short for sea-level-scenario. The scenario sets the GIS elevation raster. Scenario 1 represents elevations from a recent Digital Elevation Model at Humboldt Bay. Scenarios 2 to 4 represent projected elevation models ranging from mild sea level rise to extreme sea level rise.

Initial-number-curlews: The initial size of the curlew population at the beginning of the season.

Initial-biomass: The initial amount of generic prey biomass assigned to tidal flat patches.

Max-sight: The maximum radius that a curlew can move to when choosing a tidal flat. The radius is measured in patch lengths and is centered at the curlew's position from the last timestep

Max-weight: The maximum weight that the birds can attain. This represents a target weight and signals the bird to roost the next timestep if the target weight is reached or passed.

IFIRmax: The maximum value for Interference-Free-Intake-Rate. This is based on an estimation of the maximum intake of curlews.

## THINGS TO NOTICE

During the simulation, what  do the graphs tell us and how is this information useful? At the end of the simulation, what interesting patterns or trends do you see? Are these trends you'd expect from from the seasons?

How do changes in parameter values affect the behavior and survival of the curlews? How does changing the sea level scenario affect the presence of tidal flat patches? How about overall survival?

What do the biomass-diagnose and census-map buttons do? How can you use them in an experiment? What sort of questions can they help answer?

## THINGS TO TRY

Often, field studies and surveys of a wildlife species are difficult tasks. To get an accurate number of how many individuals of a shorebird population live in one area may require a synchronized survey of an entire area by a large number of people. An example of this is the Christmas Bird Count, a nationwide event celebrated by American bird-lovers since the year 1900. Surveys like this give us information about not only how large populations of bird species are, but also give us their distribution throughout the area and what local habitats are most important to these species. 

In the code, Part 2.5 of the go procedure is commented out. This code is used to take a snapshot of population count on three census dates in October, January, and February. These census dates coincide with bay-wide surveys done at Humboldt Bay in actual field studies. Uncomment these in the code and use them in your own experiment. For example, you can type in census1 in the command center, make a monitor button, or try outputting the census count into a spreadsheet. 

GIS: You can output a census map on each census date and upload the census map in a GIS program. Try comparing the census maps with different sea level scenarios. Refer to the GIS extension in the manual for help. 

Try creating some histograms! For example, try making a histogram of curlew weights, or a histogram of the biomass of tide patches. How do these histograms change over time?
Try subsetting the patches or agents into groups, e.g. medium, low, and high groups for patches based on elevation, or curlews by their dominance. For example, do higher dominance curlews really survive longer given our assumptions? 
The Behavior Space Tool (Ctrl+Shift+B) is an extremely useful feature of NetLogo. Play with the available experiments or create your own. Can you create some interesting graphs in Excel or R?

## EXTENDING THE MODEL

Do you have a shorebird species and a certain site that you're interested in applying the model to? Try using your own species and using parameters based on your research. A GIS map can represent the elevations of patches, but GIS research may also offer other interesting data that are useful for representing important habitat features such as availability of different prey types and their abundance, or the presence of preferred feeding or roosting areas.
This model is not limited to shorebirds in tidal habitats. With enough tinkering around, you can apply the basic ideas of the model to other types of forager-prey systems.

Good Luck with your modeling! And remember to have fun!



## CREDITS AND REFERENCES

This model was developed and used for research in this paper, available through Humboldt Digital Scholar: 
Evangelista, A.G. 2015. Modeling the Effects of Sea Level Rise on Long-billed Curlews at Humboldt Bay. Master's thesis, Humboldt State University.


Theory and assumptions are derived from the following papers and other similar research referenced in my thesis paper bibliography:
Stillman, R. A. 2008. MORPH -an individual-based model to predict the effect of environmental change on foraging animal populations. Ecological Modelling, 216:265–276.

Stillman, R. A., A. D. West, J. D. Goss-Custard, S. McGrorty, N. J. Frost, D. J. Morrisey, A. J. Kenny, and A. L. Drewitt. 2005. Predicting site quality for shorebird communities: a case study on the Humber estuary, UK. Marine Ecology Progress Series, 305:203–217.

Durell, S. E. A. L. d., R. A. Stillman, R. W. G. Caldow, S. McGrorty, A. D. West, and
J. Humphreys. 2006. Modelling the effect of environmental change on shorebirds: A case study on Poole Harbour, UK. Biological Conservation, 131:459–473.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

curlew
false
9
Polygon -6459832 true false 6 114 25 100 104 74 114 65 121 60 129 60 134 62 138 66 139 73 139 88 136 106 136 116 144 119 178 119 257 137 287 143 293 149 240 158 185 174 143 172 137 145 169 121 288 169 244 168 196 189 134 180 111 137 106 125 111 90 105 84 58 94 32 101 14 109
Polygon -6459832 true false 188 184 225 204 227 214 149 219 123 237 110 241 146 216 218 209 176 184
Polygon -6459832 true false 169 182 182 217 181 274 175 273 176 216 162 178
Polygon -13791810 true true 95 284 105 276 138 269 177 269 210 273 224 283 213 290 177 295 144 295 107 292
Circle -16777216 true false 120 68 4

curlewbackup
false
9
Polygon -6459832 true false 6 114 25 100 104 74 114 65 121 60 129 60 134 62 138 66 139 73 139 88 136 106 136 116 144 119 178 119 257 137 287 143 293 149 240 158 185 174 143 172 137 145 169 121 288 169 244 168 196 189 134 180 111 137 106 125 111 90 105 84 58 94 32 101 14 109
Polygon -6459832 true false 188 184 225 204 227 214 149 219 123 237 110 241 146 216 218 209 176 184
Polygon -6459832 true false 169 182 182 217 181 274 175 273 176 216 162 178
Polygon -13791810 true true 95 284 105 276 138 269 177 269 210 273 224 283 213 290 177 295 144 295 107 292
Circle -16777216 true false 120 68 4

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.5
@#$#@#$#@
setup
set grass? true
repeat 75 [ go ]
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="calib1d" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2200"/>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="show-energy?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="curlew-metabolism">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="curlew-food-depletion">
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-gain-from-tide">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="tide-amplitude">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-tide-height">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-biomass">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="curlews-gain-from-food">
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="SAdessic" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="720"/>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="curlew-food-depletion">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-gain-from-tide">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-biomass">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="curlews-gain-from-food">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="curlew-metabolism">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="tide-amplitude">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-tide-height">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="varymaxweight" repetitions="3" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="initial-biomass">
      <value value="15000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="max-weight" first="600" step="50" last="1000"/>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-sight">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="varymaxweightmidbiomass" repetitions="3" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="initial-biomass">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="max-weight" first="600" step="50" last="1000"/>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-sight">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="varymaxsight" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="initial-biomass">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="750"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <steppedValueSet variable="max-sight" first="1" step="1" last="10"/>
  </experiment>
  <experiment name="varymaxcurlewmidbiomass" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="initial-biomass">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="750"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="max-curlews" first="7" step="1" last="12"/>
    <enumeratedValueSet variable="max-sight">
      <value value="6"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="varymaxsightlowbiomass" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="initial-biomass">
      <value value="14000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="750"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <steppedValueSet variable="max-sight" first="1" step="1" last="10"/>
  </experiment>
  <experiment name="varyinitbiomass" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <metric>census1</metric>
    <metric>census2</metric>
    <metric>census3</metric>
    <enumeratedValueSet variable="max-sight">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="initial-biomass" first="15000" step="500" last="21000"/>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="203"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="850"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="varyinitbiomassfollowup" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <metric>census1</metric>
    <metric>census2</metric>
    <metric>census3</metric>
    <enumeratedValueSet variable="max-sight">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-biomass">
      <value value="34000"/>
      <value value="54000"/>
      <value value="58000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="850"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="varyinitbiomass-censusyr2" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <metric>census1</metric>
    <metric>census2</metric>
    <metric>census3</metric>
    <steppedValueSet variable="initial-biomass" first="0" step="2000" last="36000"/>
    <enumeratedValueSet variable="max-sight">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="IFIRmax">
      <value value="6.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-feeding-depth">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="850"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="biomass-dessication">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="205"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="varyinitbiomassCalib2" repetitions="6" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <metric>census1</metric>
    <metric>census2</metric>
    <metric>census3</metric>
    <enumeratedValueSet variable="max-sight">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="initial-biomass" first="17000" step="1000" last="40000"/>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="301"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="850"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ResultsRun23k" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <metric>average-weight</metric>
    <metric>census2</metric>
    <metric>census3</metric>
    <enumeratedValueSet variable="initial-biomass">
      <value value="23000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-sight">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="850"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="301"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="IFIRmax">
      <value value="6.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="scenario">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ResultsRun18k" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <metric>average-weight</metric>
    <metric>census2</metric>
    <metric>census3</metric>
    <enumeratedValueSet variable="initial-biomass">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-sight">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="850"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="203"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="IFIRmax">
      <value value="6.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="scenario">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ResultsRun37k" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <metric>average-weight</metric>
    <metric>census2</metric>
    <metric>census3</metric>
    <enumeratedValueSet variable="initial-biomass">
      <value value="37000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-sight">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-weight">
      <value value="850"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="301"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="IFIRmax">
      <value value="6.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="scenario">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="deathcompMean" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="max-weight">
      <value value="850"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-sight">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-curlews">
      <value value="301"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="scenario">
      <value value="3"/>
      <value value="2"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="IFIRmax">
      <value value="6.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-curlews">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-biomass">
      <value value="29000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
